<?php
/**********************************************************************
    This API endpoint processes incoming client and server information
    and saves it to a MySQL database.
***********************************************************************/

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include database configuration
include '../config.php';

// Retrieve incoming data
$clientInfo = isset($_GET['clientInfo']) ? json_decode($_GET['clientInfo'], true) : [];
$clientData = isset($_GET['clientData']) ? json_decode($_GET['clientData'], true) : [];

// Log incoming data for debugging
file_put_contents('debug.log', "Incoming Client Info: " . print_r($clientInfo, true) . "\n", FILE_APPEND);
file_put_contents('debug.log', "Incoming Client Data: " . print_r($clientData, true) . "\n", FILE_APPEND);

// Check if data is received
if (empty($clientInfo) || empty($clientData)) {
    echo json_encode(['status' => 'error', 'message' => 'No data received']);
    exit;
}

// Validate clientInfo structure
$requiredKeys = ['ip_address', 'browser', 'language', 'referrer', 'mac_address', 'hard_drive_serial', 'motherboard_serial', 'cpu_serial'];
foreach ($requiredKeys as $key) {
    if (!isset($clientInfo[$key])) {
        echo json_encode(['status' => 'error', 'message' => "Missing key in clientInfo: $key"]);
        exit;
    }
}

// Prepare an SQL statement to check for existing device in the device_info table
$stmtCheck = $conn->prepare("SELECT cpu_serial, motherboard_serial, hard_drive_serial 
    FROM device_info 
    WHERE mac_address = ?");
$stmtCheck->bind_param("s", $clientInfo['mac_address']);

if (!$stmtCheck) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to prepare statement: ' . $conn->error]);
    exit;
}

$stmtCheck->execute();
$stmtCheck->bind_result($existingCpu, $existingMotherboard, $existingHardDrive);
$stmtCheck->store_result();

$deviceExists = $stmtCheck->num_rows > 0;

if ($deviceExists) {
    // Fetch existing values
    $stmtCheck->fetch();
    
    // Check if all values match
    if (
        $existingCpu === $clientInfo['cpu_serial'] && 
        $existingMotherboard === $clientInfo['motherboard_serial'] && 
        $existingHardDrive === $clientInfo['hard_drive_serial']
    ) {
        // All details match
        echo json_encode(['status' => 'success', 'message' => 'Proceed to login page.']);
    } else {
        // None of the details match
        echo json_encode(['status' => 'error', 'message' => 'Failure: Access denied']);
        
        // Proceed to insert data into client_info
        $stmtClientInfo = $conn->prepare("INSERT INTO client_info 
            (ip_address, browser, language, referrer, mac_address, 
            screen_resolution, color_depth, device_memory, cpu_cores, 
            timezone, battery_level, network_type, 
            hard_drive_serial, motherboard_serial, cpu_serial) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

        if (!$stmtClientInfo) {
            echo json_encode(['status' => 'error', 'message' => 'Failed to prepare client_info statement: ' . $conn->error]);
            exit;
        }

        // Bind parameters for client_info
        $stmtClientInfo->bind_param("sssssssssssssss", 
            $clientInfo['ip_address'],
            $clientInfo['browser'],
            $clientInfo['language'],
            $clientInfo['referrer'],
            $clientInfo['mac_address'],
            $clientData['screenResolution'],
            $clientData['colorDepth'],
            $clientData['deviceMemory'],
            $clientData['cpuCores'],
            $clientData['timezone'],
            $clientData['batteryLevel'],
            $clientData['networkType'],
            $clientInfo['hard_drive_serial'], 
            $clientInfo['motherboard_serial'], 
            $clientInfo['cpu_serial']
        );

        // Execute the client_info statement
        if ($stmtClientInfo->execute()) {
            echo json_encode(['status' => 'error', 'message' => 'Access Denied']);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Data could not be saved in client_info: ' . $stmtClientInfo->error]);
        }

        // Close the client_info statement
        $stmtClientInfo->close();
    }
} else {
    // Device does not exist
    echo json_encode(['status' => 'error', 'message' => 'Access denied']);
}

// Close the check statement
$stmtCheck->close();
$conn->close();
?>
